using System;
using System.Collections.Generic;
using System.Diagnostics.Eventing.Reader;
using System.IO;
using System.Linq;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages
{

    #region Header

    //<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
    //<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
    //<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
    //<Developers>
    //	<Developer>Russell Stephenson</Developer>
    //</Developers>
    //<SiteName>Hines OIFO</SiteName>
    //<CreationDate>12/8/2015</CreationDate>
    //<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
    //<summary>The HL7OruMessage type represents the HL7 v2.4 General Clinical Order Message(OMG). It implements the HL7ProtocolMessage type and is not meant to be inherited.</summary>

    #endregion

    /// <summary>
    /// Class HL7OruMessage
    /// </summary>
    // ReSharper disable once InconsistentNaming
    public sealed class HL7AIOruMessage : HL7ProtocolMessage
    {
        /// <summary>
        /// General Clinical Order Message(OMG)
        /// </summary>
        private const string MESSAGE_TYPE = "ORU^R01";

        private const bool RESPONSE_REQUIRED_INDICATOR = true;

        private string _msssageType;
        private char[] _delimiters;
        private char _fieldSeparator;
        private char _componentSeperator;

        private string[] _universalServiceID;

        private readonly MshSegment _mshSegment;
        private readonly ObrSegment _obrSegment;
        private List<ObxSegment> _obxSegments;



        /// <summary>
        /// Empty static constructor added to get rid of "beforefieldinit" attribute generated by compiler.
        /// </summary>
        static HL7AIOruMessage()
        {
        }

        ///<Developers>
        ///	<Developer>Russell Stephenson</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/8/2015</CreationDate>
        ///<TestCases>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///BR_114.07
        /// <summary>
        /// Constructor to load HL7 message.
        /// </summary>
        /// <param name="message">String representation of the HL7 message.</param>
        public HL7AIOruMessage(string message)
            : base(message)
        {

            if (message == null)
                throw (new ArgumentNullException("message"));

            _message = message;

            _msssageType = string.Empty;

            _messageControlID = HL7Utility.GetMessageControlID(message);
            if (string.IsNullOrEmpty(_messageControlID))
                throw (new ArgumentNullException("message", "Message Control ID is missing."));

            _delimiters = HL7Utility.ParseGetMessageDelimiters(this.Message);

            _fieldSeparator = _delimiters[0];
            _componentSeperator = _delimiters[1];

            string[] _hl7Segments = HL7Utility.ParseGetAllMessageSegments(message);

            if (_hl7Segments == null)
                throw (new ArgumentNullException("message", "Message segements are not parsable."));

            List<string> parsingErrors = new List<string>();
            _obxSegments = new List<ObxSegment>();
            foreach (var segment in _hl7Segments)
            {
                string trimedSegment = segment.Trim();

                string segmentId = HL7Utility.ParseGetSegmentID(trimedSegment);
                if (segmentId == null)
                {
                    parsingErrors.Add("Segment ID is invalid.");
                    return;
                }
                switch (segmentId)
                {
                    case "MSH":
                        if (_mshSegment == null)
                            _mshSegment = new MshSegment(trimedSegment, _delimiters, ref parsingErrors);
                        else
                        {
                            string errorMessage = string.Format("Multiple MSH segments received in message.");
                            parsingErrors.Add(errorMessage);
                        }
                        break;
                    case "OBR":
                        if(_obrSegment == null)
                            _obrSegment = new ObrSegment(trimedSegment, _fieldSeparator, _componentSeperator, ref parsingErrors);
                        else
                        {
                            string errorMessage = string.Format("Multiple OBR segments received in message.");
                            parsingErrors.Add(errorMessage);                      
                        }
                        break;
                    case "OBX":
                        ObxSegment obxSegment = new ObxSegment(trimedSegment, _fieldSeparator, _componentSeperator,
                            ref parsingErrors);
                        //Comment is common for entire test
                        //Store the comment in the obr segment so that it can be used in the test comment don't include the comment OBX segment.
                        if (obxSegment.TestTypeId == "Comment")
                        {
                            if (_obrSegment != null)
                                _obrSegment.TestComment = obxSegment.TestResultId;
                        }
                        else if (obxSegment.TestResultId == string.Empty &&
                                 ((_obrSegment != null) && (_obrSegment.TestName == "Antigen Typing")))
                        {
                            //Skip this OBX because it has an empty result and we skip empty antigen results.
                        }
                        else
                            _obxSegments.Add(obxSegment);
                        break;
                    default:
                        break;
                }
            }

            if (_mshSegment == null)
                parsingErrors.Add("MSH segment is not found");

            if (_obrSegment == null)
                parsingErrors.Add("OBR segment is not found");

            if (_obxSegments.Count == 0)
                parsingErrors.Add("No OBX segments found");

            if (parsingErrors.Count > 0)
                throw (new HL7Exception(parsingErrors.Aggregate((workingSentence, next) =>
                    next + System.Environment.NewLine + workingSentence)));
        }

        /// <summary>
        /// WriteHL7MessageBody
        /// </summary>
        /// <param name="writer"></param>
        protected override void WriteHL7MessageBody(StreamWriter writer)
        {
            if (writer == null)
                throw (new ArgumentNullException("writer"));

            writer.Write(this.Message);

        }


        ///<Developers>
        ///	<Developer>Russell Stephenson</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/08/2015</CreationDate>
        ///<TestCases>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public override bool IsResponseRequired()
        {
            return RESPONSE_REQUIRED_INDICATOR;
        }

        ///<Developers>
        ///	<Developer>Russell Stephenson</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/08/2015</CreationDate>
        ///<TestCases>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Returns the full HL7 message.
        /// </summary>
        /// <returns>string representation of HL7 message input.</returns>
        public override string GetMessage()
        {
            return Message;
        }

        ///<Developers>
        ///	<Developer>Russell Stephenson</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/08/2015</CreationDate>
        ///<TestCases>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Returns the Message Control ID associated with this message
        /// </summary>
        /// <returns>Message Control ID in MSH segment</returns>
        public override string GetMessageControlID()
        {
            return MessageControlID;
        }

        ///<Developers>
        ///	<Developer>Russell Stephenson</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/08/2015</CreationDate>
        ///<TestCases>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Overriden method to get the Message Type.
        /// </summary>
        public override string GetMessageType()
        {
            return MessageType;
        }

        ///<Developers>
        ///	<Developer>Russell Stephenson</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/08/2015</CreationDate>
        ///<TestCases>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Message Type.
        /// </summary>
        public string MessageType
        {
            get { return MESSAGE_TYPE; }
        }

        ///<Developers>
        ///	<Developer>Russell Stephenson</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/08/2015</CreationDate>
        ///<TestCases>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Full HL7 message
        /// </summary>
        public string Message
        {
            get { return _message; }
        }

        ///<Developers>
        ///	<Developer>Russell Stephenson</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/08/2015</CreationDate>
        ///<TestCases>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Message Control ID
        /// </summary>
        public string MessageControlID
        {
            get { return _messageControlID; }
            set { _messageControlID = value; }
        }

        ///<Developers>
        ///	<Developer>Russell Stephenson</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/08/2015</CreationDate>
        ///<TestCases>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Message Delimiters
        /// </summary>
        public char[] Delimiters
        {
            get { return _delimiters; }
        }


        ///<Developers>
        ///	<Developer>Russell Stephenson</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/08/2015</CreationDate>
        ///<TestCases>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// OBR Segment
        /// </summary>
        public MshSegment Msh
        {
            get { return _mshSegment; }
            //set { _mshSegment = value; }
        }

        ///<Developers>
        ///	<Developer>Russell Stephenson</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/08/2015</CreationDate>
        ///<TestCases>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// OBR Segment
        /// </summary>
        public ObrSegment Obr
        {
            get { return _obrSegment; }
            //set { _obrSegment = value; }
        }

        ///<Developers>
        ///	<Developer>Russell Stephenson</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/08/2015</CreationDate>
        ///<TestCases>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// OBX Segment
        /// </summary>
        public List<ObxSegment> ObxSegments
        {
            get { return _obxSegments; }
            set { _obxSegments = value; }
        }

        ///<Developers>
        ///	<Developer>Russell Stephenson</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/08/2015</CreationDate>
        ///<TestCases>
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// UniversalServiceID
        /// </summary>
        public string[] UniversalServiceID
        {
            get { return _universalServiceID; }
            set { _universalServiceID = value; }
        }



        #region Support Classes

        /// <summary>
        /// 
        /// </summary>
        public class MshSegment
        {
            private readonly char[] _delimiters;

            /// <summary>
            /// 
            /// </summary>
            public char FieldSeperator { get; set; }

            /// <summary>
            /// 
            /// </summary>
            public char ComponentSeparator { get; set; }

            /// <summary>
            /// 
            /// </summary>
            public string[] MSHFields { get; set; }

            /// <summary>
            /// MSH-3-sending application 
            /// </summary>
            public string SendingApplication { get; set; }

            /// <summary>
            /// MSH-4-sending facility
            /// </summary>
            public string SendingFacility { get; set; }

            /// <summary>
            /// MSH-5-receiving application
            /// </summary>
            public string ReceivingApplication { get; set; }

            /// <summary>
            /// MSH-6-receiving facility
            /// Is the same as the DivisionCode.
            /// </summary>
            public string ReceivingFacility { get; set; }

            /// <summary>
            /// MSH-7-date/time of message
            /// </summary>
            public DateTime MessageDateTime { get; set; }

            /// <summary>
            /// MSH-8-security
            /// </summary>
            public string Security { get; set; }

            /// <summary>
            /// MSH-9-message type
            /// </summary>
            public string MessageType { get; set; }

            /// <summary>
            /// MSH-10-message control ID
            /// </summary>
            public string MessageControlID { get; set; }

            /// <summary>
            /// MSH-11-processing ID
            /// </summary>
            public string ProcessingID { get; set; }

            /// <summary>
            /// MSH-12-version ID
            /// </summary>
            public string VersionID { get; set; }

            /// <summary>
            /// MSH-13-sequence number
            /// </summary>
            public string SequenceNumber { get; set; }

            /// <summary>
            /// MSH-14-continuation pointer
            /// </summary>
            public string ContinuationPointer { get; set; }

            /// <summary>
            /// MSH-15 Accept Acknowledgment Type
            /// </summary>
            public string AcceptAcknowledgmentType { get; set; }

            /// <summary>
            /// MSH-16 Application Acknowledgment Type
            /// </summary>
            public string ApplicationAcknowledgmentType { get; set; }

            /// <summary>
            /// 
            /// </summary>
            /// <param name="msh"></param>
            /// <param name="delimiters"></param>
            /// <param name="parsingErrors"></param>
            public MshSegment(string msh, char[] delimiters, ref List<string> parsingErrors)
            {
                _delimiters = delimiters;
                if (_delimiters != null)
                {
                    FieldSeperator = _delimiters[0];
                    ComponentSeparator = _delimiters[1];
                }

                if (string.IsNullOrEmpty(msh))
                {
                    parsingErrors.Add("Failed to parse MSH segment.");
                    return;
                }
               
                MSHFields = msh.Split(FieldSeperator);

                int counter = 0;
                foreach (string mshField in MSHFields)
                {
                    switch (counter)
                    {
                        case 0: //skip MSH
                            break;
                        case 1: //skip delimiters
                            break;
                        case 2: //SendingApplication (i.e. IM)
                            if (string.IsNullOrEmpty(mshField))
                            {
                                parsingErrors.Add("Sending Application Not Found");
                            }
                            else
                            {

                                if (mshField != "IM")
                                    parsingErrors.Add("Not supported Sending Application [" + mshField + "]");
                                else
                                    SendingApplication = mshField;
                            }
                            break;
                        case 3: //SendingFacility (optional)
                            if (mshField == null)
                            {
                                parsingErrors.Add("Sending Facility Not Found");
                            }
                            else
                                SendingFacility = mshField;
                            break;
                        case 4: //ReceivingApplication
                            if (string.IsNullOrEmpty(mshField))
                            {
                                parsingErrors.Add("Receiving Application Not Found");
                            }
                            else
                            {
                                ReceivingApplication = mshField;
                            }
                            break;
                        case 5: //ReceivingFacility (i.e Division Code)
                            if (string.IsNullOrEmpty(mshField))
                            {
                                parsingErrors.Add("Receiving Facility Not Found");
                            }
                            else
                                ReceivingFacility = mshField;
                            break;
                        case 6: //MessageDateTime
                            if (string.IsNullOrEmpty(mshField))
                            {
                                parsingErrors.Add("Message Date/Time Not Found");
                            }
                            else
                            {
                                try
                                {
                                    //Defect#: 229650.
                                    MessageDateTime = HL7DateFormat.ConvertHL7LongDateTime(mshField);
                                }
                                catch (Exception e)
                                {
                                    parsingErrors.Add("Message Date/Time Not Valid" + e.Message);

                                }
                            }
                            break;
                        case 7: //Security
                            if (mshField == null)
                            {
                                parsingErrors.Add("Security Not Found");
                            }
                            else
                                Security = mshField;
                            break;
                        case 8: //MessageType
                            if (string.IsNullOrEmpty(mshField))
                            {
                                parsingErrors.Add("Message Type Not Found");
                            }
                            else
                            {
                                if (mshField != "ORU^R01")
                                    parsingErrors.Add("Not supported Mesage Type [" + mshField + "]");
                                else
                                    MessageType = mshField;
                            }
                            break;
                        case 9: //MessageControlID
                            if (string.IsNullOrEmpty(mshField))
                            {
                                parsingErrors.Add("MessageControl ID Not Found");
                            }
                            else
                                MessageControlID = mshField;
                            break;
                        case 10: //ProcessingID
                            if (string.IsNullOrEmpty(mshField))
                            {
                                parsingErrors.Add("Processing ID Not Found");
                            }
                            else
                                ProcessingID = mshField;
                            break;
                        case 11: //VersionID
                            if (string.IsNullOrEmpty(mshField))
                            {
                                parsingErrors.Add("Version ID Not Found");
                            }
                            else
                            {
                                if (mshField != "2.4")
                                    parsingErrors.Add("Not supported HL7 Version [" + mshField + "]");
                                else
                                    VersionID = mshField;
                            }
                            break;
                        case 12: //SequenceNumber
                            if (string.IsNullOrEmpty(mshField))
                            {
                                parsingErrors.Add("Sequence Number Not Found");
                            }
                            else
                                SequenceNumber = mshField;
                            break;
                        case 13: //ContinuationPointer
                            if (mshField == null)
                            {
                                parsingErrors.Add("Continuation Pointer Not Found");
                            }
                            ContinuationPointer = mshField;
                            break;
                        case 14: //AcceptAcknowledgmentType
                            if (string.IsNullOrEmpty(mshField))
                            {
                                parsingErrors.Add("Accept Acknowledgment Type Not Found");
                            }
                            else
                            {
                                if (mshField != "AL")
                                    parsingErrors.Add("Not supported AcceptAcknowledgmentType [" + mshField + "]");
                                else
                                    AcceptAcknowledgmentType = mshField;
                            }
                            break;
                        case 15: //ApplicationAcknowledgmentType
                            if (string.IsNullOrEmpty(mshField))
                            {
                                parsingErrors.Add("Application Acknowledgment Type Not Found");
                            }
                            else
                            {
                                if (mshField != "NE")
                                    parsingErrors.Add("Not supported AcceptAcknowledgmentType [" + mshField + "]");
                                else
                                    ApplicationAcknowledgmentType = mshField;
                            }
                            break;
                        default:
                            parsingErrors.Add("Unexpected field found: MSH#" + counter + ", value: " + mshField + ".");
                            break;
                    }

                    counter++;
                }
                if (counter < 16)
                    parsingErrors.Add("Invalid MSH field count:" + counter + ".");

            }

        }

        /// <summary>
        /// 
        /// </summary>
        public class ObrSegment
        {

            private string _testName;

            private readonly string[] _idFields;

            /// <summary>
            /// 
            /// </summary>
            public string[] OBRFields { get; set; }

            /// <summary>
            /// 
            /// </summary>
            public int SequenceNumber { get; set; }

            /// <summary>
            /// 
            /// </summary>
            public string TestName {
                get { return _testName; }
            }

            /// <summary>
            /// 
            /// </summary>
            public string SpecimenUid { get; set; }

            /// <summary>
            /// 
            /// </summary>
            public string UnitId { get; set; }

            /// <summary>
            /// 
            /// </summary>
            public string InstrumentName { get; set; }

            /// <summary>
            /// 
            /// </summary>
            public string TestComment { get; set; }


            /// <summary>
            /// 
            /// </summary>
            /// <param name="obr"></param>
            /// <param name="fieldSeperator"></param>
            /// <param name="componentSeparator"></param>
            /// <param name="parsingErrors"></param>
            public ObrSegment(string obr, char fieldSeperator, char componentSeparator,
                ref List<string> parsingErrors)
            {

                if (string.IsNullOrEmpty(obr))
                {
                    parsingErrors.Add("Unitialized OBR Segment");
                    return;
                }

                OBRFields = obr.Split(fieldSeperator);


                if (string.IsNullOrEmpty(obr))
                {
                    parsingErrors.Add("Faild to parse OBR segment.");
                    return;
                }

                OBRFields = obr.Split(fieldSeperator);

                int counter = 0;
                foreach (string obrField in OBRFields)
                {
                    switch (counter)
                    {
                        case 0: //Skipping OBR field.
                            break;
                        case 1: //Sequence Number is used for tracking subbordinate segments.
                            if (string.IsNullOrEmpty(obrField))
                            {
                                parsingErrors.Add("OBR Sequence Number Not Valid");
                            }
                            else //Make sure to skip using the the invalid or null objects.
                            {
                                int sequenceNum = 0;
                                if (int.TryParse(obrField, out sequenceNum))
                                {
                                    SequenceNumber = sequenceNum;
                                }
                            }
                            break;
                        case 2: //Unit or Specimen Identity.
                            if (string.IsNullOrEmpty(obrField))
                            {
                                parsingErrors.Add("UnitId or SpecimenUid not found or invalid.");
                            }
                            else //Make sure to skip using the the invalid or null objects.
                            {
                                //The specimenUId and or UnitID are in the first component of the OBR 3,
                                //unless the test is a crossmatch and then the specimenUid is in the first component and the unitId is in the second.
                                //If the second component is empty we assign the first component to both the UnitID and the SpecimenUid.
                                //If the second component has a value we assign the first component to the SpecimenUid and the second to the UnitId.
                                _idFields = obrField.Split(componentSeparator);
                                if (!string.IsNullOrEmpty(_idFields[0]) && string.IsNullOrEmpty(_idFields[1]))
                                {
                                    SpecimenUid = _idFields[0];
                                    UnitId = _idFields[0];
                                    break;
                                }
                                if (!string.IsNullOrEmpty(_idFields[0]) && !string.IsNullOrEmpty(_idFields[1]))
                                {
                                    SpecimenUid = _idFields[0];
                                    UnitId = _idFields[1];
                                    break;
                                }
                                parsingErrors.Add("Unit or Specimen Identity is missing");
                            }
                            break;
                        case 3:
                            break;
                        case 4: //Test Name
                            if (string.IsNullOrEmpty(obrField))
                            {
                                parsingErrors.Add("Test Name Not Found");
                            }
                            else //Make sure to skip using the the invalid or null objects.
                                _testName = obrField;
                            break;
                        case 5:
                            break;
                        case 6:
                            break;
                        case 7:
                            break;
                        case 8:
                            break;
                        case 9:
                            break;
                        case 10:
                            break;
                        case 11:
                            break;
                        case 12:
                            break;
                        case 13:
                            break;
                        case 14:
                            break;
                        case 15:
                            break;
                        case 16:
                            break;
                        case 17:
                            break;
                        case 18: //Instrument Name
                            if (string.IsNullOrEmpty(obrField))
                            {
                                parsingErrors.Add("Instrument Name Not Found");
                            }
                            else //Make sure to skip using the the invalid or null objects.
                                InstrumentName = obrField;
                            break;
                        default:
                            parsingErrors.Add("Unexpected field found: OBR#" + counter + ", value: " + obrField +
                                              ".");
                            break;
                    }

                    counter++;
                }
                if (counter < 19)
                    parsingErrors.Add("Invalid OBR field count:" + counter + ".");

            }
        }

        /// <summary>
        /// 
        /// </summary>
        public class ObxSegment
        {

            /// <summary>
            /// 
            /// </summary>
            public string[] ObxFields { get; private set; }

            /// <summary>
            /// 
            /// </summary>
            public int SequenceNumber { get; private set; }

            /// <summary>
            /// 
            /// </summary>
            public string TestName { get; protected set; }

            /// <summary>
            /// 
            /// </summary>
            public string TestTypeId { get; set; }

            /// <summary>
            /// 
            /// </summary>
            public char SpecimenActionCode { get; private set; }

            /// <summary>
            /// 
            /// </summary>
            public DateTime TestDateTime { get; private set; }

            /// <summary>
            /// 
            /// </summary>
            public string TestResultId { get; private set; }


            /// <summary>
            /// 
            /// </summary>
            public string TestTechId { get; private set; }

            /// <summary>
            /// 
            /// </summary>
            public Guid OrderedTestGuid { get; set; }

            /// <summary>
            /// 
            /// </summary>
            public Guid PatientSpecimenGuid { get; set; }

            /// <summary>
            /// 
            /// </summary>
            public int OrderableTestId { get; set; }

            /// <summary>
            /// OrderStatusCode
            /// </summary>
            public string OrderStatusCode { get; set; }

            /// <summary>
            /// LastUpdateUser
            /// </summary>
            public Byte[] RowVersion { get; set; }


            /// <summary>
            /// 
            /// </summary>
            /// <param name="obx"></param>
            /// <param name="fieldSeperator"></param>
            /// <param name="componentSeparator"></param>
            /// <param name="parsingErrors"></param>
            public ObxSegment(string obx, char fieldSeperator, char componentSeparator,
                ref List<string> parsingErrors)
            {

                if (string.IsNullOrEmpty(obx))
                    parsingErrors.Add("Unitialized OBX Segment");

                ObxFields = obx.Split(fieldSeperator);


                if (string.IsNullOrEmpty(obx))
                    parsingErrors.Add("Failed to parse OBX segment.");

                ObxFields = obx.Split(fieldSeperator);

                int counter = 0;
                foreach (string obxField in ObxFields)
                {
                    switch (counter)
                    {
                        case 0: //Skip this.
                            break;
                        case 1: //The sequence number.
                            if (string.IsNullOrEmpty(obxField))
                            {
                                parsingErrors.Add("OBX Sequence Number Not Valid");
                            }
                            else //Make sure to skip using the the invalid or null objects.
                            {
                                int sequenceNum = 0;
                                if (int.TryParse(obxField, out sequenceNum))
                                {
                                    SequenceNumber = sequenceNum;
                                }
                            }
                            break;
                        case 2: //Skip this.
                            break;
                        case 3: //TestTypeId is the name that is mapped to the test type identifier.
                            if (string.IsNullOrEmpty(obxField))
                            {
                                parsingErrors.Add("Test Type ID Not Found");
                            }
                            else
                                TestTypeId = obxField;
                            break;
                        case 4: //Skip this.
                            break;
                        case 5: //Test Result Id (ie. 0,1,2,A,Pos,,)
                            if (obxField == null)
                            {
                                parsingErrors.Add("OBX Test Result ID Not Found");
                            }
                            else
                                //We trim + here.
                                TestResultId = obxField.TrimEnd('+');
                            break;
                        case 6: //Skip this.
                            break;
                        case 7: //Skip this.
                            break;
                        case 8: //Skip this.
                            break;
                        case 9: //Skip this.
                            break;
                        case 10: //Skip this.
                            break;
                        case 11: //Specimen Action Code.
                            if (string.IsNullOrEmpty(obxField))
                            {
                                parsingErrors.Add("OBX Observation Result Status Not Found");
                            }
                            else if (obxField.Length != 1)
                            {
                                parsingErrors.Add(string.Format("OBX Observation Result Status: {0}, Invalid Length: {1}.", obxField, obxField.Length));                               
                            }
                            else
                            {
                                SpecimenActionCode = obxField.ToCharArray()[0];
                            }
                            break;
                        case 12: //Skip this.
                            break;
                        case 13: //Skip this.
                            break;
                        case 14: //Date Time of the test.
                            if (string.IsNullOrEmpty(obxField))
                            {
                                parsingErrors.Add("Test Date/Time Not Found");
                            }
                            else
                            {
                                try
                                {
                                    //Defect#: 229650.
                                    TestDateTime = HL7DateFormat.ConvertHL7LongDateTime(obxField);
                                }
                                catch (Exception e)
                                {
                                    parsingErrors.Add("OBX: DateTime is invalid." + e.Message);
                                }

                            }
                            break;
                        case 15: //Skip this.
                            break;
                        case 16: //Test Teck Id. 
                            if (string.IsNullOrEmpty(obxField))
                            {
                                parsingErrors.Add("Test Tech ID Not Found");
                            }
                                //We don't check for length, we rely on DB lookup and expect one record returned.
                            else
                                TestTechId = obxField;
                            break;
                        default:
                            parsingErrors.Add("Unexpected field found: #" + counter + ", value: " + obxField + ".");
                            break;
                    }

                    counter++;
                }
                if (counter < 17)
                    parsingErrors.Add("Invalid OBX field count:" + counter + ".");

            }
        }

        #endregion
    }

}


